package parser;

/**
 * Factory for N grammar non-terminal objects.
 */
public class NFact extends ATVFactory {
    /**
     * Initializer lambda for this factory.
     */
    private ILambda _initializer = new ILambda() {
        public Object apply(Object param) {
            // change state to no-op
            _initializer = NoOpLambda.Singleton;

            // nothing to initialize
            return null;
        }
    };

    /**
     * Constructor for the N factory.
     *
     * @param tkz tokenizer to use
     */
    public NFact(ITokenizer tkz) {
        super(tkz);
    }

    /**
     * Make the visitor.
     */
    private void initialize() {
        _initializer.apply(null);
    }

    /**
     * Make node.
     *
     * @param host number  token
     * @param inp  lambda to apply
     * @return produced node
     */
    private Object makeNode(NumToken host, Object inp) {
        ILambda l = (ILambda) inp;
        return l.apply(new N(host));
    }

    /**
     * Make a token visitor to parse an N non-terminal.
     *
     * @return token visitor
     */
    public ITokVisitor makeVisitor() {
        initialize();
        return new NumToken.INumVisitor() {
            public Object numCase(NumToken host, Object inp) {
                return makeNode(host, inp);
            }

            public Object defaultCase(AToken host, Object param) {
                throw new IllegalArgumentException("Wrong token: '" + host + "'");
            }
        };
    }

    /**
     * Make a token visitor that delegates to the given visitor in a chain of responsibility.
     *
     * @param successor visitor to serve as successor in the chain
     */
    public ITokVisitor makeChainedVisitor(final ITokVisitor successor) {
        initialize();
        // We have a concrete token here, so we can determine a match immediately.
        // Therefore, we do not need to set up a lambda.
        return new NumToken.INumVisitor() {
            public Object numCase(NumToken host, Object inp) {
                return makeNode(host, inp);
            }

            public Object defaultCase(AToken host, Object param) {
                return host.execute(successor, param);
            }
        };
    }

    /**
     * Make a token visitor that delegates to the given visitor in a chain of responsibility.
     *
     * @param successor       visitor to serve as successor in the chain
     * @param successorLambda lambda to pass to the successor in the default case
     */
    public ITokVisitor makeBacktrackChainedVisitor(final ITokVisitor successor, final ILambda successorLambda) {
        initialize();
        // We have a concrete token here, so we can determine a match immediately.
        // Therefore, we do not need to set up a lambda.
        return new NumToken.INumVisitor() {
            public Object numCase(NumToken host, Object inp) {
                return makeNode(host, inp);
            }

            public Object defaultCase(AToken host, Object param) {
                return host.execute(successor, successorLambda);
            }
        };
    }
}

